<?php

namespace App\Repositories;

use App\Models\Notification;
use App\Models\OpdDiagnosis;
use App\Models\OpdPatientDepartment;
use Exception;
use Filament\Notifications\Notification as FilamentNotification;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;

/**
 * Class OpdDiagnosisRepository
 *
 * @version September 8, 2020, 11:46 am UTC
 */
class OpdDiagnosisRepository extends BaseRepository
{
    /**
     * @var array
     */
    protected $fieldSearchable = [
        'opd_patient_department_id',
        'report_type',
        'report_date',
        'description',
    ];

    /**
     * Return searchable fields
     */
    public function getFieldsSearchable(): array
    {
        return $this->fieldSearchable;
    }

    /**
     * Configure the Model
     **/
    public function model()
    {
        return OpdDiagnosis::class;
    }

    public function store(array $input)
    {
        try {
            /** @var OpdDiagnosis $opdDiagnosis */
            $opdDiagnosis = $this->create($input);
            if (isset($input['file']) && ! empty($input['file'])) {
                $opdDiagnosis->addMedia($input['file'])->toMediaCollection(
                    OpdDiagnosis::OPD_DIAGNOSIS_PATH,
                    config('app.media_disk')
                );
            }
        } catch (\Exception $e) {
            throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }

    public function updateOpdDiagnosis(array $input, int $opdDiagnosisId)
    {
        try {
            /** @var OpdDiagnosis $opdDiagnosis */
            $opdDiagnosis = $this->update($input, $opdDiagnosisId);
            if (isset($input['file']) && ! empty($input['file'])) {
                $opdDiagnosis->clearMediaCollection(OpdDiagnosis::OPD_DIAGNOSIS_PATH);
                $opdDiagnosis->addMedia($input['file'])->toMediaCollection(
                    OpdDiagnosis::OPD_DIAGNOSIS_PATH,
                    config('app.media_disk')
                );
            }
            if ($input['avatar_remove'] == 1 && isset($input['avatar_remove']) && ! empty($input['avatar_remove'])) {
                removeFile($opdDiagnosis, OpdDiagnosis::OPD_DIAGNOSIS_PATH);
            }
        } catch (\Exception $e) {
            throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }

    public function deleteOpdDiagnosis(int $opdDiagnosisId)
    {
        try {
            /** @var OpdDiagnosis $opdDiagnosis */
            $opdDiagnosis = $this->find($opdDiagnosisId);
            $opdDiagnosis->clearMediaCollection(OpdDiagnosis::OPD_DIAGNOSIS_PATH);
            $this->delete($opdDiagnosisId);
        } catch (\Exception $e) {
            throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }

    public function createNotification(array $input)
    {
        try {
            $patient = OpdPatientDepartment::with('patient.user')->where(
                'id',
                $input['opd_patient_department_id']
            )->first();
            addNotification([
                Notification::NOTIFICATION_TYPE['OPD Diagnosis'],
                $patient->patient->user_id,
                Notification::NOTIFICATION_FOR[Notification::PATIENT],
                $patient->patient->user->full_name . ' your OPD diagnosis has been created.',
            ]);
        } catch (Exception $e) {
            FilamentNotification::make()
                ->danger()
                ->title($e->getMessage())
                ->send();
            // throw new UnprocessableEntityHttpException($e->getMessage());
        }
    }
}
